package JavaPonies;
/*
 * Desktop Ponies for Java
 * Written by Jirugameshi
 * 
 * Adapted from Desktop Ponies by Random Anonymous Pony (http://www.ponychan.net/chan/g/res/7033.html)
 * Ponies created by various authors
 * 
 * Inanimacy is Pinkie Pie!
 * Thanks to ryder356 for OSX testing
 */

/*
 * This is the main form that will be created on application start
 */


import java.io.*;
import java.awt.*;
import javax.swing.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.image.BufferedImage;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.Scanner;
import javax.imageio.ImageIO;


@SuppressWarnings("serial")
public class WinMain extends JFrame implements ActionListener {
	// First two numbers of version should match the Desktop Ponies version for which the features are the same
	// i.e. if we have all the features of Desktop Ponies 1.10, the version should be v1.10.xxxx
	// the minor version is just for small bug fixes specifically related to the Java port
	private static String version = "v1.25.0";

	// ge will be used to retrieve the list of monitors 
	public static GraphicsEnvironment ge; 
	// screens_to_use contains the GraphicsDevice objects representing the monitors to use
	public static GraphicsDevice[] screens_to_use;
	// MoveTimer performs the ticks for Pony movement and behavior selection
	public static Timer MoveTimer = null;
	// Static reference to the instance of WinMain
	public static WinMain currentInstance = null;
	// Random number generator
	public static Random rand = new Random();

	
	// Ponies selection panel
	public JPanel poniesPanel = new JPanel();
	
	// Start button reference
	private JButton btnStart = null; 
	
	// List of active Pony objects
	public List<Pony> Active_Ponies  = new LinkedList<Pony>();
	
	// List of loaded Pony objects
	public List<Pony> Selectable_Ponies = new LinkedList<Pony>();
	
	// List of active Effects
	public List<Effect_Form> Active_Effects = new LinkedList<Effect_Form>();
	
	// Start flag
	public static boolean Ponies_Have_Launched = false;
	
	// Last time audio was played
	public Date Audio_Last_Played = new Date();
	
	
	// AllowedMoves enum
	public enum AllowedMoves {
		None,
		Horizontal_Only,
		Vertical_Only,
        Horizontal_Vertical,
        Diagonal_Only,
        Diagonal_Horizontal,
        Diagonal_Vertical,
        All,
        MouseOver,
        Sleep
	}
	
	// Directions enum
	public enum Directions {
		top,
		bottom,
		left,
		right,
		bottom_right,
		bottom_left,
		top_right,
		top_left,
		center,
		random,
		random_not_center
	}
	
	// Behavior options
	public static final int BO_name = 1;
	public static final int BO_probability = 2;
	public static final int BO_max_duration = 3;
	public static final int BO_min_duration = 4;
	public static final int BO_speed = 5;
	public static final int BO_right_image_path = 6;
	public static final int BO_left_image_path = 7;
	public static final int BO_movement_type = 8;
	public static final int BO_linked_behavior = 9;
	public static final int BO_speaking_start = 10;
	public static final int BO_speaking_end = 11;
	public static final int BO_skip = 12;
	public static final int BO_xcoord = 13;
	public static final int BO_ycoord = 14;
	public static final int BO_object_to_follow = 15;

    // Interaction_Parameters
    public static int IP_Name = 0;
    public static int IP_Initiator = 1;
    public static int IP_Probability = 2;
    public static int IP_Proximity = 3;
    public static int IP_Target_List = 4;
    public static int IP_Target_Selection_Option = 5;
    public static int IP_Behavior_List = 6;
    public static int IP_Repeat_Delay = 7;
	
	// PROGRAM START POINT
	public static void main(String args[]) {
		// Create a new instance of WinMain() and set the static reference
		currentInstance = new WinMain();

		// Call load of options from Settings file
		Options.loadOptions();
	}
	
	// Constructor
	WinMain() {
		// Set title and version number
		super("Desktop Ponies for Java " + version);
		setIconImage(Toolkit.getDefaultToolkit().getImage("Gilgapony.gif"));
		
		// Initialize controls and panels
		initializeWinMain();
		
		// Load ponies
		initializePonies();
	}

	// Add a Pony selection panel
	private void Add_to_Menu(Pony pony) {
		// Initialize main and right panel 
		JPanel ponyselect = new JPanel();
		JPanel ponylabels = new JPanel();
		
		// Initialize the pony picture control
		JLabel ponypic = new JLabel();
		ponypic.setIcon(pony.picture);
		
		// Initialize the pony name and input controls
		JLabel ponyname = new JLabel(pony.Name);
		JLabel howmany = new JLabel("How many?");
		JNumField ponycount = new JNumField("1");
		
		// Add the label and input controls to the right panel
		ponylabels.setLayout(new GridLayout(4, 1));
		ponylabels.add(ponyname);
		ponylabels.add(new JLabel());
		ponylabels.add(howmany);
		ponylabels.add(ponycount);
		
		// Add the pony picture and right panel to the main panel
		ponyselect.setLayout(new GridLayout(1, 2));
		ponyselect.setBorder(BorderFactory.createLineBorder(Color.black, 1));
		ponyselect.add(ponypic);
		ponyselect.add(ponylabels);
		
		// Add the main panel to the pony selection panel
		poniesPanel.add(ponyselect);
		poniesPanel.getParent().validate(); //Redraw
		
		Selectable_Ponies.add(pony);
	}

	// Return the Directions value from the given string
	private Directions GetDirection(String setting) throws Exception {
		if (setting.trim().equalsIgnoreCase("top"))
			return Directions.top;
		if (setting.trim().equalsIgnoreCase("bottom"))
			return Directions.bottom;
		if (setting.trim().equalsIgnoreCase("left"))
			return Directions.left;
		if (setting.trim().equalsIgnoreCase("right"))
			return Directions.right;
		if (setting.trim().equalsIgnoreCase("bottom_right"))
			return Directions.bottom_right;
		if (setting.trim().equalsIgnoreCase("bottom_left"))
			return Directions.bottom_left;
		if (setting.trim().equalsIgnoreCase("top_right"))
			return Directions.top_right;
		if (setting.trim().equalsIgnoreCase("top_left"))
			return Directions.top_left;
		if (setting.trim().equalsIgnoreCase("center"))
			return Directions.center;
		if (setting.trim().equalsIgnoreCase("any"))
			return Directions.random;
		if (setting.trim().equalsIgnoreCase("any_notcenter"))
			return Directions.random_not_center;
		
		// If not a valid direction, throw excepion
		throw new Exception("Invalid placement direction or centering for effect.");
	}

	// Run the Desktop Ponies
	private void go_Click() {
	    // We first load interactions to get a list of names 
	    // that each pony should interact with.
	    // Latter, we "initialize" interactions to get
	    // a list of each pony object.
	    try {
	        if (Options.Interactions_Enabled)
	            Load_Interactions();
	    } catch(Exception ex) {
	        System.out.println("Unable to load interactions.  Details: " + ex.getMessage());
	    }
	    
		// Make sure we have some screens to use!
		if (screens_to_use.length == 0) {
			screens_to_use = ge.getScreenDevices();
		}
		
		// Retrieve the number of ponies to display from the pony selection panel
		List<Integer> number_of_ponies = new LinkedList<Integer>();
		int Total_Ponies = 0;
		
		// Loop through all of the pony selection panels
		for (Component ponypanel : poniesPanel.getComponents()) {
			Component controls = ((JPanel)ponypanel).getComponent(1); // Retrieve the right panel
			Component ponycount = ((JPanel)controls).getComponent(3); // Retrieve the input control
			JTextField textbox = (JTextField)ponycount; // Cast to JTextField
			
			// Get the pony count
			try {
				int numbertoadd = Integer.parseInt(textbox.getText());
				number_of_ponies.add(numbertoadd);
				Total_Ponies += numbertoadd;
			} catch(NumberFormatException ex) {
				number_of_ponies.add(0);
			}
		}
		
		// Verify if we have no ponies (omg!)
		if (Total_Ponies == 0) {
			JOptionPane.showMessageDialog(this, "The total is... no ponies...  That's TOO FEW PONY.");
			return;
		}
	
		// Make sure we don't have too many ponies
		if (Total_Ponies > Options.Max_Pony_Counter) {
			JOptionPane.showMessageDialog(this, "Sorry, I don't care who you are! " + Total_Ponies + " ponies is too much pony!  You'd only hurt yourself.\nNo, seriously your computer would crash.  Try less than " + Options.Max_Pony_Counter + " total.\n(Or override this limit in the options window)");
			return;
		}
		
		// Warn if we have a big crowd
		if (Total_Ponies > 50) {
			JOptionPane.showMessageDialog(this, Total_Ponies + " ponies!?!  I hope you can handle that much pony!  Here we go...");
		}
		
		// List of unselected ponies (the poor things)
		List<Pony> ponies_to_remove = new LinkedList<Pony>();
		
		// Loop through the selected ponies
		for (int i = 0; i < number_of_ponies.size(); i++) {
			if (number_of_ponies.get(i) == 0) {
				// Set unselected ponies to be removed
				ponies_to_remove.add(Active_Ponies.get(i));
			} else if (number_of_ponies.get(i) > 1) {
				try {
					// Duplicate any ponies with more than 1 selected
					for (int z = 1; z < number_of_ponies.get(i); z++) {
						Pony new_pony = Active_Ponies.get(i).Duplicate();
						Active_Ponies.add(new_pony);
					}
				} catch(Exception ex) {
					// Debug output
					System.out.println("Error duplicating pony: " + Active_Ponies.get(i).Name + "\n" + ex.getMessage());
				}
			}
		}
		
		// Remove the unselected ponies
		for (Pony pony : ponies_to_remove) {
			Active_Ponies.remove(pony);
		}
		
		// Initialize interactions
		try {
			if (Options.Interactions_Enabled) {
				Initialize_Interactions();
			}
		} catch(Exception ex) {
			// Debug output
			JOptionPane.showMessageDialog(this, "Unable to initialize interactions.  Details: " + ex.getMessage());
		}
		
		Pony_Startup();
	}

	private void Pony_Startup() {
		// Start the simulation
		MoveTimer.start();
		
		// Hide this window
		this.setVisible(false);
		
		// HURRAY!
		Ponies_Have_Launched = true;
	}

	// Initialize the window's components and member variables
	private void initializeWinMain() {
		// Retrieve the GraphicsEnvironment reference
		ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
		// Initialize the list of monitors to use (all)
		screens_to_use = ge.getScreenDevices();
		
		// Initialize the main timer
		MoveTimer = new Timer(30, this);
		MoveTimer.stop();
		//MoveTimer.setActionCommand("MoveTimer");
		
		// Close the application if X is pressed
		this.setDefaultCloseOperation(EXIT_ON_CLOSE);
		
		// Panel for buttons
		JPanel buttonPanel = new JPanel();
		
		// Initialize the main menu buttons
		btnStart = new JButton("GIVE ME PONIES!");
		JButton btnZeroAll = new JButton("0 all ponies");
		JButton btnOptions = new JButton("OPTIONS");
		btnStart.setActionCommand("Start");
		btnStart.addActionListener(this);
		btnZeroAll.setActionCommand("ZeroAll");
		btnZeroAll.addActionListener(this);
		btnOptions.setActionCommand("Options");
		btnOptions.addActionListener(this);
		
		// Add the buttons to the button panel
		buttonPanel.add(btnZeroAll);
		buttonPanel.add(btnOptions);
		buttonPanel.add(btnStart);

		// Initialize the layout managers
		this.setLayout(new BorderLayout());
		poniesPanel.setLayout(new ModifiedFlowLayout());
		
		// Set the pony selection panel to be scrollable vertically
		JScrollPane js = new JScrollPane(poniesPanel);
		js.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
		
		// Add the pony selection panel and buttons to the window
		this.add(js, BorderLayout.CENTER);
		this.add(buttonPanel, BorderLayout.SOUTH);
		
		// Display the window
		this.setSize(708, 537);
		this.setLocationRelativeTo(null);
		this.setVisible(true);
	}
	
	// Load pony data from the directory and display the selection panels
	private void initializePonies() {
		// Start by getting a list of directories in running path
		File dir = new File("./");
	
		// Filter for only directories
		FileFilter fileFilter = new FileFilter() {
		    public boolean accept(File file) {
		        return file.isDirectory();
		    }
		};
		// Retrieve list of directories
		File[] files = dir.listFiles(fileFilter);
	
		// Warn if no folders found
		if (files.length == 0)
			JOptionPane.showMessageDialog(this, "Error: Couldn't find any subfolders! Are you running from the right folder?\n" + dir.getAbsolutePath());
		
		String smissingponies = "Error: No pony.ini configuration file found for these folders: ";
		
		// Loop through every directory
		for (File ponydir : files) {
			// Find the config file by filtering for Pony.ini
			FilenameFilter configfilter = new FilenameFilter() {
			    public boolean accept(File dir, String name) {
			        return name.equalsIgnoreCase("pony.ini");
			    }
			};
			// Retrieve the list of Pony.ini files (should be 0 or 1)
			File[] configfile = ponydir.listFiles(configfilter);
	
			// Only continue if the config file was found
			if (configfile.length > 0) {
				Scanner fi = null;
				
				// Attempt to open the Pony.ini file
				try {
					// Determine if we are dealing with a UTF-16 encoded file
					if (IsUTFFile(configfile[0].getPath()))
						fi = new Scanner(new InputStreamReader(new FileInputStream(configfile[0].getPath()), "UTF-16"));
					else
						fi = new Scanner(new FileInputStream(configfile[0].getPath()));
				}
				catch (Exception e) {
					// For debug purposes
					JOptionPane.showMessageDialog(this, "Unable to open config file: " + configfile[0].getPath() + "\n" + e.toString());
					System.out.println("Unable to open config file: " + configfile[0].getPath());
					System.out.println(e.toString());
				}
				
				// Initialize pony variables
				String PonyName = "";
				List<Speaking_Line> PonyLines = new LinkedList<Speaking_Line>();
				List<String> Behaviors = new LinkedList<String>();
				List<String> Effects = new LinkedList<String>();

				// Read pony.ini file
				while(fi.hasNextLine()) {
					// Retrieve line from file
					String line = fi.nextLine();
					
					// Ignore empty lines and lines that start with a single quote
					if (line.length() != 0 && !line.startsWith("'")) {
						// Split the values in the line (CSV)
						String[] columns = SplitWithQualifiers(line, ",", "\"");
						
						// Make sure we have at least one value to interpret
						if (columns.length > 0) {
							if (columns[0].equalsIgnoreCase("name")) { // Set pony name
								PonyName = columns[1].trim();
							} else if (columns[0].equalsIgnoreCase("behavior")) { // Add behavior line
								Behaviors.add(line);
							} else if (columns[0].equalsIgnoreCase("speak")) { // Add pony speak line
								// 1 line name
								// 2 line text
								// 3 line sound file
								// 4 skip for normal use
								// OR
								// 1 line text
								switch (columns.length) {
									case 2:
	                                    PonyLines.add(new Speaking_Line("Unnamed", columns[1].replace("\"", ""), "", "", false));
										break;
									case 5:
	                                    PonyLines.add(new Speaking_Line(columns[1].trim(), columns[2].replace("\"", ""), ponydir.getPath() + ponydir.getPath().charAt(1), columns[3].replace("\"", "").trim(), Boolean.parseBoolean(columns[4].trim())));
	                                    break;
	                                default:
	                                 //   System.out.println("Invalid 'speak' line in pony.ini file for pony named " + PonyName + ":\n" + line + "\n" +
	                                 //           "Line must contain a name for the entry, the text to be displayed, optional: soundfile, true if entry is for a specific behavior and should be skipped normally");
	                                	break;
								}
							} else if (columns[0].equalsIgnoreCase("effect")) {
								Effects.add(line);
							}
						}
					}
				}
				// Close the file
				fi.close();
	
				// Verify that the pony has a valid name
				if (PonyName.length() > 0) {
					// Create a new pony object
					Pony newPony = new Pony(PonyName, ponydir.getPath() + ponydir.getPath().charAt(1), new Pony_Form(), PonyLines);
					
					// Loop through behaviors list
					for (String behavior : Behaviors) {
						// CSV split
						String[] columns = SplitWithQualifiers(behavior, ",", "\"");
						
						// Verify if we have an unexpected number of columns
						// || (columns.length > BO_linked_behavior && columns.length <= BO_object_to_follow))
						if (columns.length <= BO_movement_type)  {
							// Debug output
							JOptionPane.showMessageDialog(this, "Warning:  You are missing a required parameter for pony " + PonyName + " in behvaior:\n" + behavior);
						} else {
							// Initialize behavior variables
							AllowedMoves movement = AllowedMoves.None;
							String linked_behavior = "";
							String speak_start = "";
							String speak_end = "";
							int xcoord = 0;
							int ycoord = 0;
							String follow = "";
							
							boolean skip = false;
							
							// Set allowed movement
							if (columns.length > 0) {
								if (columns[BO_movement_type].trim().equalsIgnoreCase("none")) {
									movement = AllowedMoves.None;
								} else if (columns[BO_movement_type].trim().equalsIgnoreCase("horizontal_only")) {
									movement = AllowedMoves.Horizontal_Only;
								} else if (columns[BO_movement_type].trim().equalsIgnoreCase("vertical_only")) {
									movement = AllowedMoves.Vertical_Only;
								} else if (columns[BO_movement_type].trim().equalsIgnoreCase("horizontal_vertical")) {
									movement = AllowedMoves.Horizontal_Vertical;
								} else if (columns[BO_movement_type].trim().equalsIgnoreCase("diagonal_only")) {
									movement = AllowedMoves.Diagonal_Only;
								} else if (columns[BO_movement_type].trim().equalsIgnoreCase("diagonal_horizontal")) {
									movement = AllowedMoves.Diagonal_Horizontal;
								} else if (columns[BO_movement_type].trim().equalsIgnoreCase("diagonal_vertical")) {
									movement = AllowedMoves.Diagonal_Vertical;
								} else if (columns[BO_movement_type].trim().equalsIgnoreCase("all")) {
									movement = AllowedMoves.All;
								} else if (columns[BO_movement_type].trim().equalsIgnoreCase("mouseover")) {
									movement = AllowedMoves.MouseOver;
								} else if (columns[BO_movement_type].trim().equalsIgnoreCase("sleep")) {
									movement = AllowedMoves.Sleep;
								}
							}						
							
							// Fix the filenames case for images under linux
							File[] imgfiles = ponydir.listFiles();
							for (File imgcase : imgfiles) {
								if (imgcase.getName().equalsIgnoreCase(columns[BO_left_image_path].trim()))
									columns[BO_left_image_path] = imgcase.getName();
								else if (imgcase.getName().equalsIgnoreCase(columns[BO_right_image_path].trim()))
									columns[BO_right_image_path] = imgcase.getName();
							}
							
							
							// Verify if we have special behaviors
							if (columns.length > BO_linked_behavior) {
								linked_behavior = columns[BO_linked_behavior].trim();
								speak_start = columns[BO_speaking_start].trim();
								speak_end = columns[BO_speaking_end].trim();
								skip = Boolean.parseBoolean(columns[BO_skip].trim());
								xcoord = Integer.parseInt(columns[BO_xcoord].trim());
								ycoord = Integer.parseInt(columns[BO_ycoord].trim());
								 						 
								if (columns.length > BO_object_to_follow) {
									follow = columns[BO_object_to_follow].trim();
								}
								 								
							}
	
							// Add behavior values to the Pony object
	                        //                    name,     , Probability, Max_Secs  , Min_Secs  , Speed     , image path, left image path, move_type, Linked behavior, speaking line_start, speaking line_end , skip normally unless processing links, x coord, ycoord, object to follow						
							newPony.Add_Behavior(columns[BO_name].trim(), Double.parseDouble(columns[BO_probability].trim()), Double.parseDouble(columns[BO_max_duration].trim()), Double.parseDouble(columns[BO_min_duration].trim()), Double.parseDouble(columns[BO_speed].trim()), ponydir.getPath() + ponydir.getPath().charAt(1) + columns[BO_right_image_path].trim(), ponydir.getPath() + ponydir.getPath().charAt(1) + columns[BO_left_image_path].trim(), movement, linked_behavior, speak_start, speak_end, skip, xcoord, ycoord, follow);
							
							// If the pony doesn't have a picture yet
							if (newPony.picture == null) {
								// Load a static image from this behavior
								BufferedImage image = null;
								try {
									image = ImageIO.read(new File(ponydir.getPath() + ponydir.getPath().charAt(1) + columns[BO_right_image_path].trim()));
								} catch(Exception e) {
									// Debug output
									System.out.println("Could not load image: " + ponydir.getPath() + ponydir.getPath().charAt(1) + columns[6].trim());
								}
								
								if (image != null) {
									// Set the pony's picture
									newPony.picture = new ImageIcon(image);
								}
							}
						}
					}
					
					// Loop through Effects list
					for (String effect : Effects) {
						// CSV split
						String[] columns = SplitWithQualifiers(effect, ",", "\"");
						
	                    //1 = effect name
	                    //2 = behavior name
	                    //3 = right image
						//4 = left image
	                    //5 = duration
	                    //6 = delay before next
	                    //7 = location relative to pony, right
	                    //8 = center of effect, right
						//9 = location relative to pony, left
						//10 = center of effect, left
	                    //11 = effect follows pony
						
						boolean found_behavior = false;
						
						// Try to find the behavior to associate with
						for (Behavior behavior : newPony.Behaviors) {
							if (behavior.Name.equalsIgnoreCase(columns[2].replace('"', ' ').trim())) {
								Directions direction_right = Directions.center;
								Directions centering_right = Directions.center;
								Directions direction_left = Directions.center;
								Directions centering_left = Directions.center;
								
								try {
									direction_right = GetDirection(columns[7]);
									centering_right = GetDirection(columns[8]);
									direction_left = GetDirection(columns[9]);
									centering_left = GetDirection(columns[10]);
								} catch (Exception ex) {
									// Debug output
									System.out.println("Invalid placement direction or centering for effect " + columns[1] + " for pony " + PonyName + ":\n" + effect);
								}
								
								// Fix the filenames case for images under linux
								File[] imgfiles = ponydir.listFiles();
								for (File imgcase : imgfiles) {
									if (imgcase.getName().equalsIgnoreCase(columns[3].trim()))
										columns[3] = imgcase.getName();
									if (imgcase.getName().equalsIgnoreCase(columns[4].trim()))
										columns[4] = imgcase.getName();
								}
								
						        // This is where we load the animation image
								Image rightimage = null;
								Image leftimage = null;
								try {
									rightimage = Toolkit.getDefaultToolkit().createImage(ponydir.getPath() + ponydir.getPath().charAt(1) + columns[3].trim());
									leftimage = Toolkit.getDefaultToolkit().createImage(ponydir.getPath() + ponydir.getPath().charAt(1) + columns[4].trim());
								} catch(Exception e) {
									// Debug output
									System.out.println("Could not load image: " + ponydir.getPath() + ponydir.getPath().charAt(1) + columns[3].trim());
									System.out.println(e.getMessage());
								}
								
								// Add the effect to the behavior if the image loaded correctly
								if (rightimage != null && leftimage != null) {
									behavior.AddEffect(columns[1].replace('"', ' ').trim(), new ImageIcon(rightimage), new ImageIcon(leftimage), Double.parseDouble(columns[5].trim()), Double.parseDouble(columns[6].trim()), direction_right, centering_right, direction_left, centering_left, Boolean.parseBoolean(columns[11].trim()));
									found_behavior = true;
								}
								break;
							}
						}
						
						if (!found_behavior) {
							// Debug output
							System.out.println("Could not find behavior for effect " + columns[1] + " for pony " + PonyName + ":\n" + effect);
						}
					}
					
					// Link special behaviors once done loading
					newPony.Link_Behaviors();
					
					// Make sure the pony has at least one correct behavior
					if (newPony.picture != null) {
						// Add the Pony object to the list of active ponies
						Active_Ponies.add(newPony);
						
						// Create a selection panel for this Pony
						Add_to_Menu(newPony);
	
						// Debug output
						System.out.println(PonyName + " successfully loaded.");
					} else {
						// Display an error message for this pony
						JOptionPane.showMessageDialog(this, "The pony " + PonyName + " could not be loaded. Make sure the Pony.ini file has all the required information.");
					}
				}
			} else {
				smissingponies = smissingponies + "\n" + ponydir.getPath();
			}
		}
		
		if (smissingponies.length() > 64) {
			JOptionPane.showMessageDialog(this, smissingponies + "\nThese ponies were not loaded.");
		}
		
		// Verify if we have no ponies (the horror!!)
		if (Active_Ponies.isEmpty() == true) {
			JOptionPane.showMessageDialog(this, "Sorry, but you don't seem to have any ponies installed.  There should have at least been a 'Derpy' folder in the same spot as this program.");
			// Disable the start button (it would just immediately quit anyway)
			btnStart.setEnabled(false);
		}		
	}

	// Determine if this is a UTF file
	private boolean IsUTFFile(String filepath) throws Exception {
		// The idea is simple:
		// scan the file in UTF-16 until we find the pony name line
		Scanner fi = new Scanner(new InputStreamReader(new FileInputStream(filepath), "UTF-16"));
		
		while (fi.hasNextLine()) {
			String line = fi.nextLine();
			
			// If we have the name, we have a valid UTF-16 encoding
			if (line.trim().toLowerCase().startsWith("name") || line.trim().toLowerCase().startsWith("'interactionname")) {
				fi.close();
				return true;
			}
		}
		
		// If we scanned the whole file and didn't find the name, we have a regular encoding
		fi.close();
		return false;
	}
	
	// Initialize interactions
	private void Initialize_Interactions() {
		// Loop through each pony and initialize interactions
		for (Pony pony : Active_Ponies) {
			pony.Initialize_Interactions();
		}
	}

	// Load the list of interactions between ponies
	private void Load_Interactions() {
		// Open the interactions.ini file if present
		File dir = new File("./");
		// Find the config file by filtering for interactions.ini
		FilenameFilter configfilter = new FilenameFilter() {
		    public boolean accept(File dir, String name) {
		        return name.equalsIgnoreCase("interactions.ini");
		    }
		};
		// Retrieve the list of Pony.ini files (should be 0 or 1)
		File[] configfile = dir.listFiles(configfilter);
	
		// Only continue if the config file was found
		if (configfile.length > 0) {
			Scanner fi = null;
			
			// Attempt to open the interactions.ini file
			try {
				// Determine if we are dealing with a UTF-16 encoded file
				if (IsUTFFile(configfile[0].getPath()))
					fi = new Scanner(new InputStreamReader(new FileInputStream(configfile[0].getPath()), "UTF-16"));
				else
					fi = new Scanner(new FileInputStream(configfile[0].getPath()));
			}
			catch (Exception e) {
				// For debug purposes
				JOptionPane.showMessageDialog(this, "Unable to open config file: " + configfile[0].getPath() + "\n" + e.toString());
				System.out.println("Unable to open config file: " + configfile[0].getPath());
				System.out.println(e.toString());
			}
		
			while(fi.hasNextLine()) {
				String line = fi.nextLine();
				
				// Ignore empty lines and lines that start with a single quote
				if (line.length() != 0 && !line.startsWith("'")) {
					// Split the values in the line (CSV)
					String[] columns = SplitWithQualifiers(line, ",", "{", "}");
	
					// Pony found flag
					boolean ponyfound = false;
					
					// Look for the pony to interact with
					for (Pony pony : Active_Ponies) {
						try {
							String ponyname = SplitWithQualifiers(columns[IP_Initiator], ",", "\"")[0];
							
							if (pony.Name.trim().equalsIgnoreCase(ponyname.trim())) {
								ponyfound = true;
								
								int repeat_delay = 60;
								
								if (columns.length >= IP_Behavior_List) {
									repeat_delay = Integer.parseInt(columns[IP_Repeat_Delay]);
								}
								
								// If found, add the interaction
			                    pony.add_Interaction(columns[IP_Name],
	                                    ponyname,
	                                    Double.parseDouble(columns[IP_Probability]),
	                                    columns[IP_Proximity],
	                                    columns[IP_Target_List],
	                                    columns[IP_Target_Selection_Option],
	                                    columns[IP_Behavior_List],
	                                    repeat_delay);
								
							}
						} catch (Exception ex) {
							// Debug output
							System.out.println("Error loading interaction for Pony: " + pony.Name + "\n" + ex.getMessage());
						}
					}
					
			        if (!ponyfound)
			        	System.out.println("Warning:  Interaction specifies a non-existant pony: " + line);
				}
			}
		}
	}

	// Main timer tick
	private void moveTimer_Tick() {
		// List of ponies to remove in case of grave errors
		List<Pony> Ponies_to_Remove = new LinkedList<Pony>();
		
		// Loop through the active ponies
		for (Pony pony : Active_Ponies) {
			// Is the pony animation window invisible?
			if (pony.Form.isVisible() == false) {
				// Display the pony animation window
				try {
					pony.Form.setName(pony.Name);
					pony.Form.setVisible(true);
					pony.Form.Pony_Image.setDoubleBuffered(Options.EnableDoubleBuffering);
					pony.Form.setup_addpony_menu();
					
					pony.Teleport();
				} catch (Exception ex) {
					Ponies_to_Remove.add(pony);
				}
			} else {
				// If already visible, perform normal pony things
				pony.Update();
			}
		}
		
		// Remove closed ponies
		for (Pony pony : Ponies_to_Remove) {
			for (Effect_Form effect : pony.Active_Effects) {
				effect.setVisible(false);
			}
			Active_Ponies.remove(pony);
		}
		
		List<Effect_Form> effects_to_remove = new LinkedList<Effect_Form>();
		
		// Remove active effects that have run their course
		for (Effect_Form Effect : Active_Effects) {
			if ((Effect.End_Time.getTime() - (new Date()).getTime()) <= 0) {
				Effect.setVisible(false);
				effects_to_remove.add(Effect);
			}
		}
		
		// Also remove them from the list of active effects
		for (Effect_Form effect : effects_to_remove) {
			Active_Effects.remove(effect);
		}
		
		// If all ponies are gone, the program should quit
		if (Active_Ponies.isEmpty()) {
			System.exit(0);
		}
	}

	// Display the options dialog
	private void options_Click() {
		new Options(this);
	}

	// Zero all of the pony selection values
	private void zero_ponies_button_click() {
		for (Component ponypanel : poniesPanel.getComponents()) {
			Component controls = ((JPanel)ponypanel).getComponent(1);
			Component ponycount = ((JPanel)controls).getComponent(3);
			JTextField textbox = (JTextField)ponycount;
			textbox.setText("0");
		}
	}

	// actionPerformed handler (buttons and main timer)
	public void actionPerformed(ActionEvent e) {
		if (e.getSource() == MoveTimer) {
			moveTimer_Tick();
		} else if (e.getActionCommand().equals("Start")) {
			go_Click();
		} else if (e.getActionCommand().equals("Options")) {
			options_Click();
		} else if (e.getActionCommand().equals("ZeroAll")) {
			zero_ponies_button_click();			
		}
	}

	// SplitWithQualifiers function
	// Splits the string based on textdelimiter, but not if it is between textqualifiers
	public String[] SplitWithQualifiers(String SourceText, String TextDelimiter, String TextQualifier) {
		return SplitWithQualifiers(SourceText, TextDelimiter, TextQualifier, "");
	}
	public String[] SplitWithQualifiers(String SourceText, String TextDelimiter, String TextQualifier, String ClosingTextQualifier) {
		String[] strTemp;
		String[] strRes; int I; int J; String A; String B; boolean blnStart = false;
		B = "";
		
		if (TextDelimiter != " ") SourceText = SourceText.trim();
		if (ClosingTextQualifier.length() > 0) SourceText = SourceText.replace(ClosingTextQualifier, TextQualifier);
		strTemp = SourceText.split(TextDelimiter);
		for (I = 0; I < strTemp.length; I++) {
		    J = strTemp[I].indexOf(TextQualifier, 0);
		    if (J > -1) {
		        A = strTemp[I].replace(TextQualifier, "").trim();
		        String C = strTemp[I].replace(TextQualifier, "");
		        if (strTemp[I].trim().equals(TextQualifier + A + TextQualifier)) {
		                B = B + A + " \n";
		                blnStart = false;
		        } else if (strTemp[I].trim().equals(TextQualifier + C + TextQualifier)) {
	                B = B + C + " \n";
	                blnStart = false;
		        } else if (strTemp[I].trim().equals(TextQualifier + A)) {
		                B = B + A + TextDelimiter;
		                blnStart = true;
		        } else if (strTemp[I].trim().equals(A)) {
		                B = B + A + TextDelimiter;
		                blnStart = false;
		        } else if (strTemp[I].trim().equals(A + TextQualifier)) {
		                B = B + A + "\n";
		                blnStart = false;
		        }
		    } else {
		        if (blnStart)
		            B = B + strTemp[I] + TextDelimiter;
		        else
		            B = B + strTemp[I] + "\n";
		    }
		}
		if (B.length() > 0) {
		    B = B.substring(0, B.length());
		    strRes = B.split("\n");
		} else {
		    strRes = new String[1];
		    strRes[0] = SourceText;
		}
		return strRes;
	}
	
	public void sleep_all() {
		if (MoveTimer.isRunning()) {
			MoveTimer.stop();
			
			for (Pony pony : Active_Ponies) {
				pony.sleep();
			}
		} else {
			for (Pony pony : Active_Ponies) {
				pony.wake_up();
			}
			
			MoveTimer.start();
		}
	}
}